package app.composablegenerator.repo;

import com.google.cloud.storage.*;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Minimal Google Cloud Storage helper without Spring beans.
 * Uses ADC (Application Default Credentials). Bucket hardcoded per request.
 */
public final class GcsStorage {
    private static final Storage storage = StorageOptions.getDefaultInstance().getService();
    private static final String BUCKET = "composablestart"; // per user request

    private GcsStorage() {}

    public static class UploadResult {
        public final String bucket;
        public final String objectPath;
        public final long size;
        public final String gsUri;
        public UploadResult(String bucket, String objectPath, long size) {
            this.bucket = bucket;
            this.objectPath = objectPath;
            this.size = size;
            this.gsUri = "gs://" + bucket + "/" + objectPath;
        }
    }

    public static UploadResult uploadCaseFile(String caseId, String originalName, InputStream in, String contentType) throws IOException {
        String safeName = sanitize(originalName);
        String objectPath = "decisionCEO/" + caseId + "/" + safeName;
        byte[] bytes = in.readAllBytes();
        if (contentType == null || contentType.isBlank()) contentType = "application/octet-stream";

        Map<String, String> meta = new HashMap<>();
        meta.put("uploadedAt", Instant.now().toString());
        meta.put("caseId", caseId);

        BlobInfo info = BlobInfo.newBuilder(BlobId.of(BUCKET, objectPath))
                .setContentType(contentType)
                .setMetadata(meta)
                .build();
        storage.create(info, bytes);
        return new UploadResult(BUCKET, objectPath, bytes.length);
    }

    public static URL signedUrl(String objectPath, int seconds) {
        BlobInfo info = BlobInfo.newBuilder(BUCKET, objectPath).build();
        return storage.signUrl(info, seconds, TimeUnit.SECONDS, Storage.SignUrlOption.withV4Signature());
    }

    public static URL signedUrlForGsUri(String gsUri, int seconds) {
        Parsed p = parseGsUri(gsUri);
        BlobInfo info = BlobInfo.newBuilder(p.bucket, p.object).build();
        return storage.signUrl(info, seconds, TimeUnit.SECONDS, Storage.SignUrlOption.withV4Signature());
    }

    public static Path downloadToTmp(String gsUri) throws IOException {
        Parsed p = parseGsUri(gsUri);
        Blob blob = storage.get(p.bucket, p.object);
        if (blob == null) throw new IOException("Object not found: " + gsUri);
        Path target = Path.of("/tmp", p.object);
        Files.createDirectories(target.getParent());
        blob.downloadTo(target);
        return target;
    }

    private record Parsed(String bucket, String object) {}

    private static Parsed parseGsUri(String gsUri) {
        if (gsUri == null || !gsUri.startsWith("gs://")) throw new IllegalArgumentException("Invalid gs uri: " + gsUri);
        String rest = gsUri.substring(5);
        int slash = rest.indexOf('/');
        if (slash <= 0) throw new IllegalArgumentException("Invalid gs uri: " + gsUri);
        String bucket = rest.substring(0, slash);
        String object = rest.substring(slash + 1);
        return new Parsed(bucket, object);
    }

    private static String sanitize(String name) {
        if (name == null || name.isBlank()) return "file.bin";
        return name.replace("..", ".").replace("\\", "/").replace('/', '_');
    }
}
