package app.composablegenerator.repo;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.firestore.CollectionReference;
import com.google.cloud.firestore.Firestore;
import com.google.cloud.firestore.FirestoreOptions;

// Minimal logging without external deps

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

public final class FirestoreFactory {
    private FirestoreFactory() {}

    // Config for this app
    private static final String DATABASE_ID = "composablestart"; // target database as requested
    private static final String SERVICE_ACCOUNT_RESOURCE = "composableai-firestore.json"; // under src/main/resources
    // Namespace prefix for all collections of this app
    private static final String NAMESPACE = "ontologyMap"; // collection prefix, e.g., ontologyMap_concepts

    public static Firestore create() {
        // Log relevant environment and intent
        String envProject = orEmpty(System.getenv("GOOGLE_CLOUD_PROJECT"));
        String envEmulator = orEmpty(System.getenv("FIRESTORE_EMULATOR_HOST"));
        String envAdc = orEmpty(System.getenv("GOOGLE_APPLICATION_CREDENTIALS"));
        log("Initializing Firestore client. Env GOOGLE_CLOUD_PROJECT=" + envProject +
                ", FIRESTORE_EMULATOR_HOST=" + envEmulator +
                ", GOOGLE_APPLICATION_CREDENTIALS=" + envAdc +
                "; Using classpath service account if provided.");

        try {
            GoogleCredentials credentials = loadCredentialsFromClasspath(SERVICE_ACCOUNT_RESOURCE);
            FirestoreOptions.Builder b = FirestoreOptions.newBuilder()
                    .setDatabaseId(DATABASE_ID)
                    .setCredentials(credentials);
            if (!envProject.isEmpty()) {
                b.setProjectId(envProject);
            }
            FirestoreOptions options = b.build();

            Firestore service = options.getService();
            // Summarize resolved options
            String summary = "Firestore ready: projectId=" + options.getProjectId()
                    + ", databaseId=" + options.getDatabaseId()
                    + ", host=firestore.googleapis.com:443 (forced prod)";
            log(summary);

            return service;
        } catch (IOException e) {
            logErr("Failed to load credentials from: " + SERVICE_ACCOUNT_RESOURCE, e);
            throw new RuntimeException("Failed to load Firestore credentials from resource: " + SERVICE_ACCOUNT_RESOURCE, e);
        } catch (RuntimeException e) {
            logErr("Firestore initialization failed: " + e.getMessage(), e);
            throw e;
        }
    }

    // Return a namespaced collection reference, e.g., ontologyMap_concepts
    public static CollectionReference collection(String baseName) {
        String coll = NAMESPACE + "_" + baseName;
        return create().collection(coll);
    }

    private static GoogleCredentials loadCredentialsFromClasspath(String resourceName) throws IOException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        if (cl == null) cl = FirestoreFactory.class.getClassLoader();
        URL url = cl.getResource(resourceName);
        if (url == null) {
            throw new IOException("Resource not found on classpath: " + resourceName);
        }
        log("Credential resource located at: " + url);

        try (InputStream is = cl.getResourceAsStream(resourceName)) {
            if (is == null) {
                throw new IOException("Resource stream is null for: " + resourceName);
            }
            log("Credential resource stream opened. Parsing credentials...");
            GoogleCredentials creds = GoogleCredentials.fromStream(is);
            log("Credential file parsed successfully.");
            return creds;
        }
    }

    private static String orEmpty(String s) { return s == null ? "" : s; }

    private static void log(String msg) { System.out.println("[FirestoreFactory] " + msg); }
    private static void logErr(String msg, Throwable t) {
        System.err.println("[FirestoreFactory] ERROR: " + msg);
        if (t != null) t.printStackTrace();
    }
}
