Developer Guide – Lean Layered, POJO‑First Java Architecture

Overview
This document explains the project’s architecture and shows concrete examples of how to build views, handle JSON, and use the core services (LLMHandler, Whatsapp, SimpleLogger). The goal is to run an Java Spring App on google cloud with Firestore as database. Firebase Storage for files.  The folder includes files for: model.md, view.md and control.md to describe the app. It also includes ai.md for the AI part and build.md for a step by step build approach. Build one step each and then ask for feedback or do testing! The views are constructed with HTML building blocks mainly header, main and footer based on tailwind / daisy. Examples are provided.


Philosophy (short)
- Controller → Service → Repo/Client: one job per layer, minimal magic.
- POJO‑first models: getters/setters only; map to JSON and Firestore.
- Explicit data flow: predictable request → service → persistence.
- Configuration via profiles/env; avoid hardcoding credentials in code.

Standard structure (repo specifics)
- `controller`: HTTP endpoints and simple HTML views.
- `service`: orchestration and business rules (no HTTP/DB here).
- `repo`: Firestore CRUD (no business rules here).
- `util`: view rendering and JSON helpers.
- `model`: DTOs/POJOs only.

Request flow pattern
- Controller receives the request, validates inputs, calls Service.
- Service loads/saves through Repo and calls external clients.
- Controller renders a JSON or HTML view.

Creating views with ViewHelper
ViewHelper assembles HTML from fragments with minimal templating.
- Compose a page from fragments:
  - `ViewHelper.composePage("header.html", "mainuser.html", "footer.html", Map.of())`.
- Render a fragment with a model map:
  - `String body = ViewHelper.render(ViewHelper.loadFragment("main_message.html"), Map.of("message", txt))`.
- Fragment discovery order: `classpath:/templates/*`, `classpath:/static/*`, then filesystem.

Example: simple HTML endpoint (from UserController)
- `@GetMapping(value = "/singleUser", produces = MediaType.TEXT_HTML_VALUE)`
- Build the page: `ViewHelper.composePage("header.html", "mainuser.html", "footer.html", Map.of())`.
- In the `mainuser.html` script, call backend APIs (`/api/v1/user/...`) as needed.
- NO nesting URLs! Keep all user endpoints under root for easy CSS and JS template handling.

Passing data into templates
- Provide a model map with keys used as `{{key}}` placeholders in fragments.
- Example (from UserInteractions):
  - Build the model: `Map.of("client_name", name, "message", finalMsg)`
  - Render: `String main = ViewHelper.render(ViewHelper.loadFragment("main_message.html"), model)`

Working with JSON via util classes
Use lightweight wrappers around Jackson for ergonomic JSON building/parsing.

- Build JSON:
  - `JSONObject obj = new JSONObject().put("name", "Alex").put("active", true);`
  - `JSONArray arr = new JSONArray().put("run").put("rest");`
  - Nesting: `obj.put("plan", new JSONObject().put("days", 3)); obj.put("tags", arr);`

- Parse JSON:
  - `JSONObject in = new JSONObject(rawString);`
  - Accessors: `in.getString("field")`, `in.getInt("n")`, `in.getBoolean("ok")`.
  - Nested: `in.getJSONObject("meta")`, `in.getJSONArray("items")`.
  - Safe get: `Object any = in.get("field")` (returns boxed primitives or nested wrappers).

- Serialize:
  - `String json = obj.toString();` or `arr.toString();`

Example: LLM context JSON (see UserInteractions)
- Build `JSONObject`+`JSONArray` for the prompt context, then pass `context.toString()` to LLMHandler.

Services: LLMHandler, Whatsapp, SimpleLogger

LLMHandler (text generation)
- Typical call (from `UserInteractions`):
  - `JSONObject ai = llm.getOpenAIResponse(contextJson, "gpt-5-mini", system, "", previousId);`
  - The previousID needs to be "" if the communication is new!
- Measure generation latency:
  - `long t0 = System.currentTimeMillis(); ... call ... long t1 = System.currentTimeMillis();`
  - Persist duration and timestamps with the message metadata (see `MessageRepository.save`).
- Post‑processing:
  - Fallback if text is blank; cap length; update the user’s `lastResponseId` when present.

Whatsapp (outbound)
- Send plain text:
  - `String resp = Whatsapp.sendTextWhatsApp(user.getPhone(), "Hello there");`
- Send template:
  - `String resp = Whatsapp.sendWhatsApp(user.getPhone(), "<message>", "<template_name>");`
- Tips:
  - Do not log full payloads with PII; use `SimpleLogger.logInfo(tag, SimpleLogger.shortString(payload, N))` if needed.
  - Inject tokens via env/props; avoid hardcoding secrets in code.

SimpleLogger (structured logging)
- Info (short): `SimpleLogger.logInfo("Tag", "Message");`
- Info (long): `SimpleLogger.logInfoLong("Tag", longMessage);`
- Warning/Error: `SimpleLogger.logWarning("Tag", msg);` / `SimpleLogger.logError("Tag", msg, ex);`
- Performance: `startTrackingTime()` + `logCurrentTimeLapsed(tag, msg)` or manual timestamps around calls.
- Tags: use concise, feature‑oriented names (e.g., `UserController`, `MessageController`, `Whatsapp`).

Persistence patterns (Firestore)
- Repos hide Firestore SDK specifics and keep models clean.
- ID policy:
  - Generate new doc IDs in repo when `id` is blank; store a stable `uniqueId` on `User` docs.
- Timestamps:
  - Store Firestore `Timestamp` in DB; expose epoch millis in models.
- Queries:
  - Use `collectionGroup("messages")` to read from both top‑level and subcollections.

Example: saving an outbound message (see `UserInteractions`)
- Build `Message` with `direction=out`, `clientId`, `bodyText`, `createdAt`.
- Add metadata: `llm_started_at`, `llm_finished_at`, `llm_duration_ms`, `ai_response_id`.
- Call `MessageService.save(message)`.

Use the JSON composableai-firestore.json for authentication to Firestore and use the composablestart database for the initial launch.
Make sure you have an application name and use a dedicated namespace in the dataspace for this app for any objects!

Adding a new HTML view (checklist)
1) Create main fragment under `src/main/resources/templates/*`. Reuse `header.html` and `footer.html`.
2) Add a controller method with `produces = MediaType.TEXT_HTML_VALUE`.
3) Compose the page with `ViewHelper.composePage(...)` or render a fragment with a model.
4) If the page needs data, expose a JSON API under `/api/v1/...` and fetch it from the page script.

Testing tips
- Unit‑test services with repositories mocked or pointing to emulator/test project.
- Controllers: prefer `@WebMvcTest` for HTML/JSON endpoints; keep page scripts thin.
- Avoid logging sensitive data; prefer aggregate/short logs via `SimpleLogger`.
