package app.composablegenerator.service;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Base64;
import java.util.List;

/**
 * Minimal, self-contained LLM handler for OpenAI-compatible chat APIs.
 *
 * Notes
 * - No secrets are hardcoded; uses env vars:
 *   - OPENAI_API_KEY (required for live calls)
 *   - OPENAI_BASE_URL (default: https://api.openai.com/v1)
 *   - OPENAI_MODEL (default: gpt-5)
 * - Methods return a compact JSON result with id, model, and content.
 * - Includes 3 convenience methods to call an LLM with file context.
 */
public class LLMHandler {
    private final ObjectMapper mapper = new ObjectMapper();
    private final HttpClient http = HttpClient.newHttpClient();

    public LLMHandler() {}

    // Core method used by all helpers
    public ObjectNode getOpenAICompletions(String userContent,
                                        String model,
                                        String systemPrompt,
                                        String assistantContext,
                                        String previousId) {
        String apiKey = "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";
        String base = getenvOr("OPENAI_BASE_URL", "https://api.openai.com/v1");
        String mdl = (model == null || model.isBlank()) ? getenvOr("OPENAI_MODEL", "gpt-5") : model;

        ObjectNode result = mapper.createObjectNode();
        result.put("model", mdl);

        if (apiKey.isBlank()) {
            result.put("id", "offline-demo");
            result.put("content", "[LLM offline] " + safe(userContent));
            return result;
        }

        try {
            ObjectNode body = mapper.createObjectNode();
            body.put("model", mdl);
            ArrayNode messages = body.putArray("messages");
            if (systemPrompt != null && !systemPrompt.isBlank()) {
                ObjectNode sys = mapper.createObjectNode();
                sys.put("role", "system");
                sys.put("content", systemPrompt);
                messages.add(sys);
            }
            if (assistantContext != null && !assistantContext.isBlank()) {
                ObjectNode asst = mapper.createObjectNode();
                asst.put("role", "assistant");
                asst.put("content", assistantContext);
                messages.add(asst);
            }
            ObjectNode usr = mapper.createObjectNode();
            usr.put("role", "user");
            usr.put("content", userContent);
            messages.add(usr);

            HttpRequest req = HttpRequest.newBuilder()
                    .uri(URI.create(base + "/chat/completions"))
                    .timeout(Duration.ofSeconds(60))
                    .header("Authorization", "Bearer " + apiKey)
                    .header("Content-Type", "application/json")
                    .POST(HttpRequest.BodyPublishers.ofString(body.toString(), StandardCharsets.UTF_8))
                    .build();

            HttpResponse<String> resp = http.send(req, HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
            if (resp.statusCode() < 200 || resp.statusCode() >= 300) {
                result.put("id", "http-" + resp.statusCode());
                result.put("content", "HTTP " + resp.statusCode() + ": " + resp.body());
                return result;
            }
            JsonNode root = mapper.readTree(resp.body());
            String id = root.path("id").asText("");
            String content = root.path("choices").path(0).path("message").path("content").asText("");
            result.put("id", id.isBlank() ? "ok" : id);
            result.put("content", content);
            return result;
        } catch (Exception e) {
            result.put("id", "error");
            result.put("content", "Exception: " + e.getMessage());
            return result;
        }
    } 
    
    public String listInputItems(String responseId) {
    	  String apiKey = "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";
        String base   = getenvOr("OPENAI_BASE_URL", "https://api.openai.com/v1");
        String url    = base + "/responses/" + responseId + "/input_items";

        try {
            HttpRequest req = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .timeout(Duration.ofSeconds(30))
                .header("Authorization", "Bearer " + apiKey)
                .header("Content-Type", "application/json")
                .GET()
                .build();

            HttpResponse<String> resp = http.send(req, HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
            if (resp.statusCode() < 200 || resp.statusCode() >= 300) {
                return "HTTP " + resp.statusCode() + ": " + resp.body();
            }

            JsonNode root = mapper.readTree(resp.body());
            ArrayNode data = (ArrayNode) root.path("data");

            StringBuilder sb = new StringBuilder();
            for (JsonNode item : data) {
                String id   = item.path("id").asText();
                String role = item.path("role").asText();
                // Most items have content[0].text
                String content = "";
                JsonNode contentNode = item.path("content");
                if (contentNode.isArray() && contentNode.size() > 0) {
                    content = contentNode.get(0).path("text").asText();
                }
                sb.append(id).append(" [").append(role).append("]: ").append(content).append("\n");
            }

            return sb.toString();

        } catch (Exception e) {
            return "Exception: " + e.getMessage();
        }
    }

    
    public ObjectNode getOpenAIResponse(String userContent,
    		String model,
    		String systemPrompt,
    		String assistantContext,
    		String previousId) {
    	 String apiKey = "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";
    	String base = getenvOr("OPENAI_BASE_URL", "https://api.openai.com/v1");
    	String mdl  = (model == null || model.isBlank()) ? getenvOr("OPENAI_MODEL", "gpt-4.1-mini") : model;

    	ObjectNode result = mapper.createObjectNode();
    	result.put("model", mdl);

    	if (apiKey.isBlank()) {
    		result.put("id", "offline-demo");
    		result.put("content", "[LLM offline] " + safe(userContent));
    		return result;
    	}

    	try {
    		ObjectNode body = mapper.createObjectNode();
    		body.put("model", mdl);

    		// Build input messages (Responses API)
    		ArrayNode input = body.putArray("input");
    		if (systemPrompt != null && !systemPrompt.isBlank()) {
    			input.add(mapper.createObjectNode()
    					.put("role", "system")
    					.put("content", systemPrompt));
    		}
    		if (assistantContext != null && !assistantContext.isBlank()) {
    			input.add(mapper.createObjectNode()
    					.put("role", "assistant")
    					.put("content", assistantContext));
    		}
    		input.add(mapper.createObjectNode()
    				.put("role", "user")
    				.put("content", userContent));

    		// Threading: previous_response_id
    		if (previousId != null && !previousId.isBlank()) {
    			body.put("previous_response_id", previousId);
    		}

    		HttpRequest req = HttpRequest.newBuilder()
    				.uri(URI.create(base + "/responses"))
    				.timeout(Duration.ofSeconds(60))
    				.header("Authorization", "Bearer " + apiKey)
    				.header("Content-Type", "application/json")
    				.POST(HttpRequest.BodyPublishers.ofString(body.toString(), StandardCharsets.UTF_8))
    				.build();

    		HttpResponse<String> resp = http.send(req, HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
    		if (resp.statusCode() < 200 || resp.statusCode() >= 300) {
    			result.put("id", "http-" + resp.statusCode());
    			result.put("content", "HTTP " + resp.statusCode() + ": " + resp.body());
    			return result;
    		}

    		JsonNode root = mapper.readTree(resp.body());
    		String id = root.path("id").asText("");
    		SimpleLogger.logInfo("LLMHAndler", extractUsage(root));
    		String content = extractOutputText(root);
    		if (content.isBlank()) {
    		    // Final fallback: try to read the first message’s plain text if present
    		    JsonNode output = root.path("output");
    		    if (output.isArray() && output.size() > 0) {
    		        JsonNode firstContent = output.get(0).path("content");
    		        if (firstContent.isArray() && firstContent.size() > 0) {
    		            content = firstContent.get(0).path("text").asText("");
    		        }
    		    }
    		}
    		// Avoid dumping the entire JSON unless you explicitly want debug output
    		if (content.isBlank()) content = "[No textual output in response]";

    		result.put("id", id.isBlank() ? "ok" : id);
    		result.put("content", content);
    		return result;

    	} catch (Exception e) {
    		result.put("id", "error");
    		result.put("content", "Exception: " + e.getMessage());
    		return result;
    	}
    }
    
    private String extractOutputText(JsonNode root) {
        // Responses API: root.output is an array of items
        JsonNode output = root.path("output");
        if (!output.isArray() || output.size() == 0) return "";

        StringBuilder sb = new StringBuilder();
        for (JsonNode item : output) {
            // Most common shape: a message with a content array
            JsonNode contentArr = item.path("content");
            if (contentArr.isArray()) {
                for (JsonNode c : contentArr) {
                    String type = c.path("type").asText();
                    if ("output_text".equals(type)) {
                        String t = c.path("text").asText("");
                        if (!t.isBlank()) {
                            if (sb.length() > 0) sb.append("\n");
                            sb.append(t);
                        }
                    }
                }
            }
            // (Optional) Other shapes could appear in future (e.g., tool calls).
            // You can extend here if you need to capture those as well.
        }
        return sb.toString();
    }
    public String extractUsage(JsonNode root) {
        try {
            
            JsonNode usage = root.path("usage");

            String inputTokens  = usage.path("input_tokens").asText();
            String inputCached  = usage.path("input_tokens_details").path("cached_tokens").asText();
            String outputTokens = usage.path("output_tokens").asText();

            return "input_tokens=" + inputTokens +
                   ", input_cached=" + inputCached +
                   ", output_tokens=" + outputTokens;
        } catch (Exception e) {
            return "Exception: " + e.getMessage();
        }
    }


    // OpenAI Embeddings: returns {model, id, dims, embedding:[...]}
    public ObjectNode getOpenAIEmbedding(String text) {
        String apiKey = "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";
        String base = getenvOr("OPENAI_BASE_URL", "https://api.openai.com/v1");
        String mdl = "text-embedding-3-small";

        ObjectNode result = mapper.createObjectNode();
        result.put("model", mdl);

        if (apiKey.isBlank()) {
            result.put("id", "offline-demo");
            result.put("dims", 0);
            result.set("embedding", mapper.createArrayNode());
            return result;
        }

        try {
            ObjectNode body = mapper.createObjectNode();
            body.put("model", mdl);
            body.put("input", safe(text));

            HttpRequest req = HttpRequest.newBuilder()
                    .uri(URI.create(base + "/embeddings"))
                    .timeout(Duration.ofSeconds(60))
                    .header("Authorization", "Bearer " + apiKey)
                    .header("Content-Type", "application/json")
                    .POST(HttpRequest.BodyPublishers.ofString(body.toString(), StandardCharsets.UTF_8))
                    .build();

            HttpResponse<String> resp = http.send(req, HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
            if (resp.statusCode() < 200 || resp.statusCode() >= 300) {
                result.put("id", "http-" + resp.statusCode());
                result.put("dims", 0);
                result.set("embedding", mapper.createArrayNode());
                return result;
            }

            JsonNode root = mapper.readTree(resp.body());
            String model = root.path("model").asText(mdl);
            JsonNode embNode = root.path("data").path(0).path("embedding");
            result.put("model", model);
            result.put("id", root.path("data").path(0).path("index").asInt(0));
            // Copy to an ArrayNode to keep a consistent shape
            ArrayNode arr = mapper.createArrayNode();
            if (embNode.isArray()) {
                for (JsonNode v : embNode) {
                    arr.add(v.asDouble());
                }
            }
            result.set("embedding", arr);
            result.put("dims", arr.size());
            return result;
        } catch (Exception e) {
            ObjectNode err = mapper.createObjectNode();
            err.put("model", mdl);
            err.put("id", "error");
            err.put("dims", 0);
            err.set("embedding", mapper.createArrayNode());
            err.put("content", "Exception: " + e.getMessage());
            return err;
        }
    }

    // 1) Chat with a single local file (reads content or embeds a base64 preview)
    public ObjectNode chatWithFile(Path file,
                                   String model,
                                   String systemPrompt,
                                   String instruction) {
        try {
            String fileName = file.getFileName().toString();
            byte[] bytes = Files.readAllBytes(file);
            String preview;
            // If small text or JSON, include as text preview; otherwise include base64 header only
            if (bytes.length <= 200_000 && looksLikeText(bytes)) {
                String text = new String(bytes, StandardCharsets.UTF_8);
                preview = trimTo(text, 15_000);
            } else {
                String b64 = Base64.getEncoder().encodeToString(slice(bytes, 0, 50_000));
                preview = "[binary-preview base64 first 50KB] " + b64;
            }
            String user = "File: " + fileName + "\nInstruction: " + safe(instruction) + "\n\nPreview:\n" + preview;
            return getOpenAIResponse(user, model, systemPrompt, "", "");
        } catch (IOException e) {
            ObjectNode err = mapper.createObjectNode();
            err.put("id", "ioerror");
            err.put("content", "Failed to read file: " + e.getMessage());
            return err;
        }
    }

    // 2) Chat with a file by URL (no download here; passes URL reference)
    public ObjectNode chatWithFileUrl(String fileUrl,
                                      String model,
                                      String systemPrompt,
                                      String instruction) {
        String user = "Please analyze the file at this URL and follow the instruction.\nURL: " + safe(fileUrl) +
                "\nInstruction: " + safe(instruction);
        return getOpenAIResponse(user, model, systemPrompt, "", "");
    }

    // 3) Chat with multiple local files (concise previews)
    public ObjectNode chatWithFiles(List<Path> files,
                                    String model,
                                    String systemPrompt,
                                    String instruction) {
        StringBuilder sb = new StringBuilder();
        sb.append("Instruction: ").append(safe(instruction)).append("\n\n");
        for (Path f : files) {
            try {
                byte[] bytes = Files.readAllBytes(f);
                String preview;
                if (bytes.length <= 120_000 && looksLikeText(bytes)) {
                    preview = trimTo(new String(bytes, StandardCharsets.UTF_8), 8_000);
                } else {
                    preview = "[binary] size=" + bytes.length + " name=" + f.getFileName();
                }
                sb.append("--- ").append(f.getFileName()).append(" ---\n").append(preview).append("\n\n");
            } catch (IOException e) {
                sb.append("--- ").append(f.getFileName()).append(" ---\n<io error: ").append(e.getMessage()).append(">\n\n");
            }
        }
        return getOpenAIResponse(sb.toString(), model, systemPrompt, "", "");
    }

    private static String getenvOr(String k, String def) {
        String v = System.getenv(k);
        return (v == null || v.isBlank()) ? def : v;
    }
    private static String safe(String s) { return s == null ? "" : s; }
    private static boolean looksLikeText(byte[] bytes) {
        int n = Math.min(bytes.length, 4096);
        for (int i = 0; i < n; i++) {
            byte b = bytes[i];
            if (b == 0) return false; // likely binary
        }
        return true;
    }
    private static String trimTo(String s, int max) {
        if (s.length() <= max) return s;
        return s.substring(0, max) + "\n...[truncated]";
    }
    private static byte[] slice(byte[] in, int off, int max) {
        int end = Math.min(in.length, off + max);
        byte[] out = new byte[end - off];
        System.arraycopy(in, off, out, 0, out.length);
        return out;
    }
}
